<?php
/**
 * Plugin Name: Sentinel Security Shield
 * Plugin URI: https://sentinel.atide.tech
 * Description: Unified Security Shield for WordPress. Protects against Brute Force, Carding, and Bots using AI.
 * Version: 1.0.0
 * Author: Sentinel Security Inc.
 * License: GPL2
 */

if (!defined('ABSPATH')) exit;

class SentinelWP {
    private static $instance = null;
    private $api_key;
    private $mode;

    public static function getInstance() {
        if (self::$instance == null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        add_action('admin_menu', [$this, 'add_plugin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        
        $this->api_key = get_option('sentinel_api_key');
        $this->mode = get_option('sentinel_mode', 'hybrid');

        if ($this->api_key) {
            add_action('init', [$this, 'protect_request'], 1);
            add_action('wp_loaded', [$this, 'start_buffer']);
        }
    }

    public function start_buffer() {
        if (!is_admin()) {
            ob_start([$this, 'inject_sentinel_assets']);
        }
    }

    public function inject_sentinel_assets($buffer) {
        if (stripos($buffer, '</body>') !== false) {
            $js = '<script src="https://sentinel.atide.tech/sdk/shield.js?key=' . esc_attr($this->api_key) . '" async></script>';
            $hp = '<input type="text" name="sentinel_hp" style="display:none !important" tabindex="-1" autocomplete="off">';
            $buffer = str_replace('</form>', $hp . '</form>', $buffer);
            return str_ireplace('</body>', $js . '</body>', $buffer);
        }
        return $buffer;
    }

    public function protect_request() {
        if (is_admin() || (defined('XMLRPC_REQUEST') && XMLRPC_REQUEST)) return;

        $api_url = 'https://sentinel.atide.tech/api/shield.php';
        
        $request_data = [
            'api_key' => $this->api_key,
            'uri' => $_SERVER['REQUEST_URI'],
            'method' => $_SERVER['REQUEST_METHOD'],
            'payload' => $this->sanitize_payload($_POST),
            'metadata' => [
                'ua' => $_SERVER['HTTP_USER_AGENT'],
                'lang' => $_SERVER['HTTP_ACCEPT_LANGUAGE']
            ]
        ];

        // Simple sync/async logic for WP
        if ($this->mode === 'performance' || ($_SERVER['REQUEST_METHOD'] === 'GET' && $this->mode === 'hybrid')) {
            $this->call_sentinel_async($api_url, $request_data);
        } else {
            $decision = $this->call_sentinel_sync($api_url, $request_data);
            if (isset($decision['action']) && $decision['action'] === 'BLOCK') {
                wp_die('<h1>Accès Refusé</h1><p>Votre activité a été détectée comme suspecte par Sentinel Security.</p>', 'Sentinel Block', ['response' => 403]);
            }
        }
    }

    private function call_sentinel_sync($url, $data) {
        $response = wp_remote_post($url, [
            'body' => json_encode($data),
            'headers' => [
                'Content-Type' => 'application/json',
                'X-Sentinel-Key' => $this->api_key
            ],
            'timeout' => 2
        ]);

        if (is_wp_error($response)) return ['action' => 'ALLOW'];
        return json_decode(wp_remote_retrieve_body($response), true);
    }

    private function call_sentinel_async($url, $data) {
        wp_remote_post($url, [
            'body' => json_encode($data),
            'headers' => [
                'Content-Type' => 'application/json',
                'X-Sentinel-Key' => $this->api_key
            ],
            'timeout' => 0.1,
            'blocking' => false
        ]);
    }

    private function sanitize_payload($payload) {
        $sanitized = [];
        $sensitive_keys = ['card', 'cvv', 'pass', 'secret', 'token'];
        foreach ($payload as $key => $value) {
            $is_sensitive = false;
            foreach ($sensitive_keys as $sk) {
                if (stripos($key, $sk) !== false) {
                    $is_sensitive = true;
                    break;
                }
            }
            $sanitized[$key] = $is_sensitive ? '[REDACTED]' : $value;
        }
        return $sanitized;
    }

    public function add_plugin_menu() {
        add_menu_page('Sentinel Security', 'Sentinel', 'manage_options', 'sentinel-settings', [$this, 'settings_page'], 'dashicons-shield-alt');
    }

    public function register_settings() {
        register_setting('sentinel-settings-group', 'sentinel_api_key');
        register_setting('sentinel-settings-group', 'sentinel_mode');
    }

    public function settings_page() {
        ?>
        <div class="wrap">
            <h1>Sentinel Security Settings</h1>
            <form method="post" action="options.php">
                <?php settings_fields('sentinel-settings-group'); ?>
                <?php do_settings_sections('sentinel-settings-group'); ?>
                <table class="form-table">
                    <tr valign="top">
                        <th scope="row">Clé API</th>
                        <td><input type="text" name="sentinel_api_key" value="<?php echo esc_attr(get_option('sentinel_api_key')); ?>" class="regular-text" /></td>
                    </tr>
                    <tr valign="top">
                        <th scope="row">Mode de Protection</th>
                        <td>
                            <select name="sentinel_mode">
                                <option value="performance" <?php selected(get_option('sentinel_mode'), 'performance'); ?>>Performance (Asynchrone)</option>
                                <option value="hybrid" <?php selected(get_option('sentinel_mode'), 'hybrid'); ?>>Hybride (Bloque les POST)</option>
                                <option value="paranoid" <?php selected(get_option('sentinel_mode'), 'paranoid'); ?>>Paranoïaque (Synchrone)</option>
                            </select>
                        </td>
                    </tr>
                </table>
                <?php submit_button(); ?>
            </form>
        </div>
        <?php
    }
}

SentinelWP::getInstance();
