<?php
/**
 * Sentinel Security Module for PrestaShop
 * Version: 1.0.0
 */

if (!defined('_PS_VERSION_')) {
    exit;
}

class SentinelShield extends Module
{
    public function __construct()
    {
        $this->name = 'sentinelshield';
        $this->tab = 'front_office_features';
        $this->version = '1.0.0';
        $this->author = 'Sentinel Security Inc.';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = ['min' => '1.7', 'max' => _PS_VERSION_];
        $this->bootstrap = true;

        parent::__construct();

        $this->displayName = $this->l('Sentinel Security Shield');
        $this->description = $this->l('Protect your PrestaShop store against bots, carding, and brute-force using AI.');

        $this->confirmUninstall = $this->l('Are you sure you want to uninstall Sentinel? Your store will no longer be protected.');
    }

    public function install()
    {
        return parent::install() &&
            $this->registerHook('header') &&
            $this->registerHook('actionDispatcher') &&
            Configuration::updateValue('SENTINEL_API_KEY', '') &&
            Configuration::updateValue('SENTINEL_MODE', 'hybrid');
    }

    public function uninstall()
    {
        return parent::uninstall() &&
            Configuration::deleteByName('SENTINEL_API_KEY') &&
            Configuration::deleteByName('SENTINEL_MODE');
    }

    public function getContent()
    {
        $output = '';

        if (Tools::isSubmit('submitSentinelConfig')) {
            $apiKey = (string)Tools::getValue('SENTINEL_API_KEY');
            $mode = (string)Tools::getValue('SENTINEL_MODE');

            Configuration::updateValue('SENTINEL_API_KEY', $apiKey);
            Configuration::updateValue('SENTINEL_MODE', $mode);

            $output .= $this->displayConfirmation($this->l('Settings updated'));
        }

        return $output . $this->renderForm();
    }

    public function renderForm()
    {
        $fields_form = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Sentinel Configuration'),
                    'icon' => 'icon-cogs'
                ],
                'input' => [
                    [
                        'type' => 'text',
                        'label' => $this->l('API Key'),
                        'name' => 'SENTINEL_API_KEY',
                        'required' => true
                    ],
                    [
                        'type' => 'select',
                        'label' => $this->l('Protection Mode'),
                        'name' => 'SENTINEL_MODE',
                        'options' => [
                            'query' => [
                                ['id' => 'performance', 'name' => 'Performance (Async)'],
                                ['id' => 'hybrid', 'name' => 'Hybrid (Blocks POST)'],
                                ['id' => 'paranoid', 'name' => 'Paranoid (Sync)']
                            ],
                            'id' => 'id',
                            'name' => 'name'
                        ]
                    ]
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                    'class' => 'btn btn-default pull-right'
                ]
            ]
        ];

        $helper = new HelperForm();
        $helper->show_toolbar = false;
        $helper->table = $this->table;
        $helper->identifier = $this->identifier;
        $helper->submit_action = 'submitSentinelConfig';
        $helper->currentIndex = $this->context->link->getAdminLink('AdminModules', false) . '&configure=' . $this->name;
        $helper->token = Tools::getAdminTokenLite('AdminModules');
        $helper->fields_value['SENTINEL_API_KEY'] = Configuration::get('SENTINEL_API_KEY');
        $helper->fields_value['SENTINEL_MODE'] = Configuration::get('SENTINEL_MODE');

        return $helper->generateForm([$fields_form]);
    }

    public function hookActionDispatcher($params)
    {
        // Don't protect Admin or Webhooks
        if ($params['controller_type'] == Dispatcher::HANDLER_BACK_OFFICE) {
            return;
        }

        $apiKey = Configuration::get('SENTINEL_API_KEY');
        $mode = Configuration::get('SENTINEL_MODE');

        if (empty($apiKey)) return;

        // Extract request data
        $request_data = [
            'api_key' => $apiKey,
            'uri' => $_SERVER['REQUEST_URI'],
            'method' => $_SERVER['REQUEST_METHOD'],
            'payload' => $this->sanitize_payload($_POST),
            'metadata' => [
                'ua' => $_SERVER['HTTP_USER_AGENT'],
                'lang' => $_SERVER['HTTP_ACCEPT_LANGUAGE']
            ]
        ];

        $api_url = 'https://sentinel.atide.tech/api/shield.php';
        $report_url = 'https://sentinel.atide.tech/api/report.php';

        if ($mode === 'performance' || ($_SERVER['REQUEST_METHOD'] === 'GET' && $mode === 'hybrid')) {
            $this->call_sentinel_async($api_url, $request_data, $apiKey);
        } else {
            $decision = $this->call_sentinel_sync($api_url, $request_data, $apiKey);
            if (isset($decision['action']) && $decision['action'] === 'BLOCK') {
                die($this->l('Access Denied by Sentinel Security Shield.'));
            }
        }
    }

    public function hookHeader()
    {
        $apiKey = Configuration::get('SENTINEL_API_KEY');
        if (empty($apiKey)) return;

        // Auto-inject JS Shield and Honeypot via JS for PrestaShop (cleaner for Smarty)
        return '
        <script src="https://sentinel.atide.tech/sdk/shield.js?key=' . $apiKey . '" async></script>
        <script>
            document.addEventListener("DOMContentLoaded", function() {
                const forms = document.querySelectorAll("form");
                forms.forEach(form => {
                    if (!form.querySelector(\'input[name="sentinel_hp"]\')) {
                        const hp = document.createElement("input");
                        hp.type = "text";
                        hp.name = "sentinel_hp";
                        hp.style.display = "none";
                        hp.tabIndex = -1;
                        hp.autocomplete = "off";
                        form.appendChild(hp);
                    }
                });
            });
        </script>';
    }

    private function call_sentinel_sync($url, $data, $apiKey) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'X-Sentinel-Key: ' . $apiKey
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 2);
        
        $response = curl_exec($ch);
        if (curl_errno($ch)) return ['action' => 'ALLOW'];
        
        return json_decode($response, true) ?? ['action' => 'ALLOW'];
    }

    private function call_sentinel_async($url, $data, $apiKey) {
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'X-Sentinel-Key: ' . $apiKey
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT_MS, 100);
        curl_setopt($ch, CURLOPT_NOSIGNAL, 1);
        curl_exec($ch);
        curl_close($ch);
    }

    private function sanitize_payload($payload) {
        $sanitized = [];
        $sensitive_keys = ['card', 'cvv', 'pass', 'secret', 'token'];
        foreach ($payload as $key => $value) {
            $is_sensitive = false;
            foreach ($sensitive_keys as $sk) {
                if (stripos($key, $sk) !== false) {
                    $is_sensitive = true;
                    break;
                }
            }
            $sanitized[$key] = $is_sensitive ? '[REDACTED]' : $value;
        }
        return $sanitized;
    }
}
